import base64
import shutil
import tempfile
import time
import math
import os
import requests
from requests import HTTPError


class CollectionDeployerHelper:
    """Helper class to deploy app collections using the compute.build API.

    This class handles zipping the collection, initiating the deployment,
    and polling for the completion of the deployment.
    """
    MAX_TIMEOUT = 15  # Maximum allowed timeout in minutes
    POLL_INTERVAL = 15  # Polling interval in seconds

    def __init__(self, collection_path: str, base_url: str, api_key: str,
                 max_timeout: int, app_collection_uuid: str) -> None:
        """Initialize CollectionDeployerHelper.

        Args:
            collection_path (str): The root path of the app collection.
            base_url (str): The base URL of the API.
            api_key (str): The API key for authentication.
            max_timeout (int): The maximum time (in minutes) to wait for deployment completion.
            app_collection_uuid (str): The UUID of the app collection to deploy.

        Raises:
            ValueError: If the max_timeout is greater than the allowed maximum.
            FileNotFoundError: If the collection_path does not exist.
        """
        self._validate_args(collection_path=collection_path, max_timeout=max_timeout)
        self.base_url: str = base_url
        self.collection_path: str = collection_path
        self.api_key: str = api_key
        self.max_timeout: int = max_timeout
        self.collection_uuid: str = app_collection_uuid
        self.is_success: bool = False

    @property
    def headers(self) -> dict:
        """Get the headers for API requests.

        Returns:
            dict: Headers including the API key and environment.
        """
        return {"X-API-KEY": self.api_key, "Akt-Environ": "Production"}

    @staticmethod
    def _validate_args(collection_path: str, max_timeout: int) -> None:
        """Validate initialization arguments.

        Args:
            collection_path (str): The path to the app collection.
            max_timeout (int): The maximum timeout in minutes.

        Raises:
            ValueError: If max_timeout exceeds the allowed maximum.
            FileNotFoundError: If the collection_path does not exist.
        """
        if max_timeout > CollectionDeployerHelper.MAX_TIMEOUT:
            raise ValueError(
                f'Error parsing the arguments: The chosen max_timeout "{max_timeout}" is too high. '
                f'Please set this value to less than {CollectionDeployerHelper.MAX_TIMEOUT} min.'
            )
        if not os.path.exists(collection_path):
            raise FileNotFoundError(
                f'Error parsing the arguments: The collection folder path "{collection_path}" was not found.'
            )

    def _get_admin_op_url(self, op_uuid: str) -> str:
        """Construct the URL to get an admin operation item.

        Args:
            op_uuid (str): The admin operation UUID.

        Returns:
            str: The full URL for the admin operation.
        """
        return f"{self.base_url}/app_collections/{self.collection_uuid}/admin_ops/{op_uuid}"

    def _get_collection_cicd_deploy_url(self) -> str:
        """Construct the URL to start the collection deployment.

        Returns:
            str: The full URL to initiate deployment.
        """
        return f"{self.base_url}/app_collections/{self.collection_uuid}/cicd/collection_start_deploy"

    def _timeout_admin_op(self, op_uuid: str, msg: str) -> None:
        """Mark an admin operation as timed out (failed_client).

        Args:
            op_uuid (str): The admin operation UUID.
            msg (str): The message describing the timeout.
        """
        url = self._get_admin_op_url(op_uuid)
        data = {
            "admin_op_status": "failed_client",
            "msg_json": {"header": "deploy [failed_client]", "body": msg}
        }
        try:
            response = requests.patch(url=url, headers=self.headers, json=data)
            response.raise_for_status()
        except HTTPError as e:
            print(f"Failed to update admin op status on timeout: {e}")
        except Exception as e:
            print(f"Unexpected error updating admin op status on timeout: {e}")

    @staticmethod
    def _zip_collection(collection_path: str) -> bytes:
        """Zip the collection directory and return the archive content as bytes.

        Args:
            collection_path (str): The path to the collection to be zipped.

        Returns:
            bytes: The content of the zip archive.
        """
        with tempfile.TemporaryDirectory() as tmp_dir:
            archive_base = os.path.join(tmp_dir, "coll_archive")
            archive_path = shutil.make_archive(archive_base, "zip", collection_path)
            with open(archive_path, "rb") as f:
                zip_data = f.read()
        return zip_data

    def start_collection_deployment(self) -> str:
        """Start the collection deployment process by uploading the zipped collection.

        Returns:
            str: The admin operation UUID for the initiated collection deployment.

        Raises:
            Exception: If an error occurs during the upload.
        """
        print("----------------------------------------------------------")
        print("Starting App Collection Deployment... ")
        try:
            zip_bytes = self._zip_collection(self.collection_path)
            collection_deploy_url = self._get_collection_cicd_deploy_url()
            encoded_data = base64.b64encode(zip_bytes).decode("utf-8")
            data = {"data_base64": encoded_data, "timeout": self.max_timeout}
            response = requests.post(url=collection_deploy_url, headers=self.headers, json=data)
            response.raise_for_status()
            op_uuid: str = response.content.decode("utf-8")
            print(f"Deployment initiated with op_uuid: {op_uuid}")
            return op_uuid
        except HTTPError as e:
            exc_msg = e.response.content.decode("utf-8") if e.response.content else str(e)
            raise Exception(f"Error during Collection Upload: {exc_msg}") from e
        except Exception as e:
            raise Exception(f"Unknown Error during Collection Upload: {e}") from e

    def poll_for_completion(self, op_uuid: str) -> None:
        """Poll for the completion status of the deployment operation.

        Args:
            op_uuid (str): The admin operation UUID to poll for.

        Raises:
            Exception: If the deployment fails or an error occurs during polling.
            TimeoutError: If the deployment does not complete within the timeout period.
        """
        print("Polling for task completion...")
        max_retries = math.floor(self.max_timeout * 60 / self.POLL_INTERVAL)
        status_endpoint = self._get_admin_op_url(op_uuid)
        for i in range(max_retries):
            try:
                status_resp = requests.get(url=status_endpoint, headers=self.headers)
                status_resp.raise_for_status()
                admin_op_json = status_resp.json()
                op_status = admin_op_json.get("admin_op_status")
                op_msg = admin_op_json.get("admin_op_msg", {})
                msg_header = op_msg.get('header', 'No header')
                print(f"Check {i + 1}/{max_retries}: {msg_header}")
                if op_status == "success":
                    self.is_success = True
                    return
                elif op_status == "failed":
                    msg_body = op_msg.get("body", "Deployment failed without a message")
                    raise Exception(msg_body)
            except Exception as e:
                raise Exception(f"Error checking status: {e}") from e
            time.sleep(self.POLL_INTERVAL)
        timeout_msg = (
            "Timed out waiting for deployment completion. NOTE: It's possible the collection was still "
            "deployed to the server."
        )
        self._timeout_admin_op(op_uuid, msg=timeout_msg)
        raise TimeoutError(timeout_msg)
